//////////////////////////////////////////////
// PastCode.h
//
// Some code from last tutorials used
// to initialize and clean a bit the main
// file
//////////////////////////////////////////////

/// Includes ---------------------------------

// nkLog
#include <NilkinsLog/Loggers/ConsoleLogger.h>

// nkGraphics
#include <NilkinsGraphics/Cameras/Camera.h>
#include <NilkinsGraphics/Cameras/CameraManager.h>

#include <NilkinsGraphics/Encoders/Obj/ObjEncoder.h>

#include <NilkinsGraphics/Entities/Entity.h>

#include <NilkinsGraphics/Graph/Node.h>
#include <NilkinsGraphics/Graph/NodeManager.h>

#include <NilkinsGraphics/Log/LogManager.h>

#include <NilkinsGraphics/Meshes/Utils/MeshUtils.h>

#include <NilkinsGraphics/Meshes/Mesh.h>
#include <NilkinsGraphics/Meshes/MeshManager.h>

#include <NilkinsGraphics/Programs/Program.h>
#include <NilkinsGraphics/Programs/ProgramManager.h>
#include <NilkinsGraphics/Programs/ProgramSourcesHolder.h>

#include <NilkinsGraphics/RenderQueues/RenderQueue.h>
#include <NilkinsGraphics/RenderQueues/RenderQueueManager.h>

#include <NilkinsGraphics/Samplers/Sampler.h>
#include <NilkinsGraphics/Samplers/SamplerManager.h>

#include <NilkinsGraphics/Shaders/Memory/ConstantBuffer.h>
#include <NilkinsGraphics/Shaders/Memory/ShaderInstanceMemorySlot.h>
#include <NilkinsGraphics/Shaders/Memory/ShaderPassMemorySlot.h>

#include <NilkinsGraphics/Shaders/Shader.h>
#include <NilkinsGraphics/Shaders/ShaderManager.h>

#include <NilkinsGraphics/System.h>

#include <NilkinsGraphics/Textures/Texture.h>
#include <NilkinsGraphics/Textures/TextureManager.h>

// nkResources
#include <NilkinsResources/ResourceManager.h>

// nkWinUi
#include <NilkinsWinUi/System.h>

// Standards
#include <chrono>

#include <memory>

/// Internals : Mesh -------------------------

void prepareMeshFromFile ()
{
	// Initialize mesh to use later
	nkGraphics::Mesh* mesh = nkGraphics::MeshManager::getInstance()->createOrRetrieve("Mesh") ;

	// Load data from file
	nkMemory::String absPath = nkResources::ResourceManager::getInstance()->getAbsoluteFromWorkingDir("sphere.obj") ;
	nkMemory::Buffer objData = nkResources::ResourceManager::getInstance()->loadFileIntoMemory(absPath) ;

	// We change some settings to alter the way the mesh is imported
	nkGraphics::ObjDecodeOptions objOptions ;
	objOptions._invertUvY = true ;
	objOptions._invertWindingOrder = true ;
	nkGraphics::DecodedData objDecoded = nkGraphics::ObjEncoder::decode(objData, objOptions) ;

	// Fill mesh
	nkGraphics::MeshFillOptions fillOptions ;
	fillOptions._autoLoad = true ;
	fillOptions._dataFillType = nkGraphics::DATA_FILL_TYPE::FORWARD ;
	nkGraphics::MeshUtils::fillMeshFromDecodedData(objDecoded._meshData[0], mesh, fillOptions) ;
}

void addMeshToRenderQueue ()
{
	nkGraphics::RenderQueue* rq = nkGraphics::RenderQueueManager::getInstance()->get(nkGraphics::RenderQueueManager::DEFAULT_RENDER_QUEUE) ;

	nkGraphics::Mesh* mesh = nkGraphics::MeshManager::getInstance()->createOrRetrieve("Mesh") ;

	nkGraphics::Entity* ent = rq->addEntity() ;
	ent->setRenderInfo(nkGraphics::EntityRenderInfo(mesh, nullptr)) ;
}

void addEntityToGraph ()
{
	nkGraphics::RenderQueue* rq = nkGraphics::RenderQueueManager::getInstance()->get(nkGraphics::RenderQueueManager::DEFAULT_RENDER_QUEUE) ;
	nkGraphics::Entity* ent = rq->getEntity(0) ;

	nkGraphics::Node* node = nkGraphics::NodeManager::getInstance()->createOrRetrieve("node") ;
	node->setPositionAbsolute(nkMaths::Vector(0.f, 0.f, 10.f)) ;

	ent->setParentNode(node) ;
}

/// Internals : Texture ----------------------

void prepareSampler ()
{
	nkGraphics::Sampler* sampler = nkGraphics::SamplerManager::getInstance()->createOrRetrieve("sampler") ;
	sampler->load() ;
}

/// Internals : Shader -----------------------

void assignShader ()
{
	nkGraphics::RenderQueue* rq = nkGraphics::RenderQueueManager::getInstance()->get(nkGraphics::RenderQueueManager::DEFAULT_RENDER_QUEUE) ;
	nkGraphics::Entity* ent = rq->getEntity(0) ;
	nkGraphics::Shader* shader = nkGraphics::ShaderManager::getInstance()->get("shader") ;

	nkGraphics::Mesh* mesh = nkGraphics::MeshManager::getInstance()->createOrRetrieve("Mesh") ;
	ent->setRenderInfo(nkGraphics::EntityRenderInfo(mesh, shader)) ;
}

/// Internals : Base -------------------------

void baseInit ()
{
	// Prepare the mesh we will show
	prepareMeshFromFile() ;
	addMeshToRenderQueue() ;
	addEntityToGraph() ;

	// Prepare the texture we will use
	prepareSampler() ;
}

void renderLoop (nkGraphics::RenderContext* context)
{
	// Get some data to prepare for the rendering loop
	nkGraphics::Camera* camera = nkGraphics::CameraManager::getInstance()->getDefaultCamera() ;

	nkGraphics::System* nkGraphicsSystem = nkGraphics::System::getInstance() ;
	nkWinUi::System* nkWinUiSystem = nkGraphicsSystem->getUiSystem() ;

	const float loopTimeMs = 5 * 1000.f ;
	std::chrono::system_clock::time_point firstNow = std::chrono::system_clock::now() ;

	const nkMaths::Vector sphereCenter = nkMaths::Vector(0.f, 0.f, 10.f) ;

	// Our rendering loop is ready to start
	while (nkGraphicsSystem->getHasRunToContinue())
	{
		// Frame the graphics system
		if (!nkGraphicsSystem->frame(context))
			break ;

		// Loop the camera around the sphere
		std::chrono::system_clock::time_point now = std::chrono::system_clock::now() ;
		float currentTimeMs = std::chrono::duration_cast<std::chrono::microseconds>(now - firstNow).count() / 1000.f ;
		
		float currentLoopT = (std::fmod(currentTimeMs, loopTimeMs) / loopTimeMs) * 2.f - 1.f ;
		currentLoopT *= 3.14 ;
		
		nkMaths::Vector loopPos (std::cos(currentLoopT) * 10.f, 0.f, std::sin(currentLoopT) * 10.f) ;
		loopPos += sphereCenter ;

		camera->setPositionAbsolute(loopPos) ;
		camera->lookAt(sphereCenter, nkMaths::Vector(0.f, 1.f, 0.f)) ;

		// Tick the windowing system
		nkWinUiSystem->tick() ;
	}
}